<?php

namespace App\Http\Controllers\Apps;

use Illuminate\Http\Request;
use App\Models\BarcodeTemplate;
use App\Http\Controllers\Controller;
use App\Models\ProductVariant;

class BarcodeController extends Controller
{
    public function index(Request $request)
    {
        $currentPage = $request->input('page', 1);
        $perPage = $request->input('per_page', 10);

        $barcodeTemplates = BarcodeTemplate::query()
            ->when($request->search, function ($query, $search) {
                $query->where('name', 'like', '%' . $search . '%');
            })
            ->paginate($perPage, ['*'], 'page', $currentPage)->withQueryString();

        return inertia('apps/barcode/index', [
            'barcodeTemplates' => $barcodeTemplates,
            'currentPage' => $currentPage,
            'perPage' => $perPage,
        ]);
    }

    public function create()
    {
        return inertia('apps/barcode/create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'paper_type' => ['required', 'in:roll,sheet'],
            'paper_width_mm' => ['required', 'numeric', 'min:1'],
            'paper_height_mm' => [
                'nullable',
                'numeric',
            ],
            'label_width_mm' => ['required', 'numeric', 'min:1'],
            'label_height_mm' => ['required', 'numeric', 'min:1'],
            'columns' => [
                'nullable',
                'integer',
            ],
            'rows' => [
                'nullable',
                'integer',
            ],
        ]);

        BarcodeTemplate::create($request->all());

        return to_route('apps.barcode-templates.index');
    }

    public function edit(BarcodeTemplate $barcodeTemplate)
    {
        return inertia('apps/barcode/edit', [
            'barcodeTemplate' => $barcodeTemplate,
        ]);
    }

    public function update(Request $request, BarcodeTemplate $barcodeTemplate)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'paper_type' => ['required', 'in:roll,sheet'],
            'paper_width_mm' => ['required', 'numeric', 'min:1'],
            'paper_height_mm' => [
                'nullable',
                'numeric',
            ],
            'label_width_mm' => ['required', 'numeric', 'min:1'],
            'label_height_mm' => ['required', 'numeric', 'min:1'],
            'columns' => [
                'nullable',
                'integer',
            ],
            'rows' => [
                'nullable',
                'integer',
            ],
        ]);

        $barcodeTemplate->update($request->all());

        return to_route('apps.barcode-templates.index');
    }

    public function generate(BarcodeTemplate $barcodeTemplate)
    {
        return inertia('apps/barcode/generate', [
            'template' => $barcodeTemplate,
            'variants' => ProductVariant::with('product', 'unit' , 'product_variant_values')->get()->map(function($item){
                $variantValues = $item->product_variant_values
                    ->map(fn($v) => $v->variant_value->name)
                    ->implode(' ');

                return [
                    'id'            => $item->id,
                    'name'          => trim($item->product->name . ' ' . $variantValues),
                    'price'         => number_format($item->price, 0, ',', '.'),
                    'barcode'       => $item->barcode,
                    'unit_name'     => $item->unit->name,
                ];
            }),
        ]);
    }

    public function destroy(BarcodeTemplate $barcodeTemplate)
    {
        $barcodeTemplate->delete();

        return to_route('apps.barcode-templates.index');
    }
}
