<?php

namespace App\Exports;

use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\{
    FromCollection,
    WithHeadings,
    WithMapping,
    WithStyles
};
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class SaleProductReportExport implements FromCollection, WithHeadings, WithMapping, WithStyles
{
    protected $from;
    protected $to;
    protected $stockableType;
    protected $stockableId;

    public function __construct($from, $to, $stockableType = null, $stockableId = null)
    {
        $this->from = $from;
        $this->to   = $to;
        $this->stockableType = $stockableType;
        $this->stockableId   = $stockableId;
    }

    public function collection()
    {
        $menuQuery = DB::table('transaction_details as td')
            ->join('transactions as t', 't.id', '=', 'td.transaction_id')
            ->join('menus as m', 'm.id', '=', 'td.items_id')
            ->where('td.items_type', \App\Models\Menu::class)
            ->where('t.status', 'paid')
            ->whereBetween('t.transaction_date', [$this->from, $this->to])
            ->selectRaw("
                'menu' as item_type,
                m.id as item_id,
                m.name as item_name,
                NULL as barcode,
                NULL as unit_name,
                td.price,
                SUM(td.quantity) as total_sale,
                SUM(td.quantity * td.price) as total_revenue
            ")
            ->groupBy('m.id', 'm.name', 'td.price');

        $variantSalesSub = DB::table('transaction_details as td')
            ->join('transactions as t', 't.id', '=', 'td.transaction_id')
            ->where('td.items_type', \App\Models\ProductVariant::class)
            ->where('t.status', 'paid')
            ->whereBetween('t.transaction_date', [$this->from, $this->to])
            ->selectRaw("
                td.items_id as variant_id,
                td.price,
                SUM(td.quantity) as total_sale,
                SUM(td.quantity * td.price) as total_revenue
            ")
            ->groupBy('td.items_id', 'td.price');

        $variantQuery = DB::table(DB::raw("({$variantSalesSub->toSql()}) as vs"))
            ->mergeBindings($variantSalesSub)
            ->join('product_variants as pv', 'pv.id', '=', 'vs.variant_id')
            ->join('products as p', 'p.id', '=', 'pv.product_id')
            ->leftJoin('units as u', 'u.id', '=', 'pv.unit_id')
            ->leftJoin('product_variant_values as pvv', 'pvv.product_variant_id', '=', 'pv.id')
            ->leftJoin('variant_values as vv', 'vv.id', '=', 'pvv.variant_value_id')
            ->selectRaw("
                'product_variant' as item_type,
                pv.id as item_id,
                TRIM(
                    CONCAT(
                        p.name,
                        COALESCE(
                            CONCAT(' ', GROUP_CONCAT(vv.name ORDER BY vv.name SEPARATOR ' ')),
                            ''
                        )
                    )
                ) as item_name,
                pv.barcode as barcode,
                u.name as unit_name,
                vs.price,
                vs.total_sale,
                vs.total_revenue
            ")
            ->groupBy(
                'pv.id',
                'p.name',
                'pv.barcode',
                'u.name',
                'vs.price',
                'vs.total_sale',
                'vs.total_revenue'
            );

        if ($this->stockableType && $this->stockableId) {
            if ($this->stockableType === 'menu') {
                $menuQuery->where('m.id', $this->stockableId);
                $variantQuery->whereRaw('1 = 0');
            }

            if ($this->stockableType === 'product_variant') {
                $variantQuery->where('pv.id', $this->stockableId);
                $menuQuery->whereRaw('1 = 0');
            }
        }

        return $menuQuery
            ->unionAll($variantQuery)
            ->orderByDesc('total_sale')
            ->get();
    }

    public function headings(): array
    {
        return [
            'Nama Item',
            'Tipe Item',
            'Barcode',
            'Satuan',
            'Harga',
            'Total Terjual',
            'Total Pendapatan',
        ];
    }

    public function map($row): array
    {
        return [
            $row->item_name,
            $row->item_type === 'menu' ? 'Menu' : 'Produk',
            ' '.$row->barcode ?? '-',
            $row->unit_name ?? '-',
            (int) $row->price,
            (int) $row->total_sale,
            (int) $row->total_revenue,
        ];
    }

    public function styles(Worksheet $sheet)
    {
        return [
            1 => ['font' => ['bold' => true]],
        ];
    }
}
